/*!
 * @file        apm32l0xx_misc.c
 *
 * @brief       This file provides all the miscellaneous firmware functions (add-on to CMSIS functions)
 *
 * @version     V1.0.0
 *
 * @date        2021-12-25
 *
 * @attention
 *
 *  Copyright (C) 2020-2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be usefull and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

#include "apm32l0xx_misc.h"

/** @addtogroup Peripherals_Library Standard Peripheral Library
  @{
*/

/** @addtogroup MISC_Driver MISC Driver
  @{
*/

/** @addtogroup MISC_Fuctions Fuctions
  @{
*/

/** @defgroup MISC_Private_Functions
  @{
 */

/*!
 * @brief       Enable NVIC request
 *
 * @param       irq:        The NVIC interrupt request, detailed in IRQn_Type
 *
 * @param       priority:   Specifies the priority needed to set
 *
 * @retval      None
 */
void NVIC_EnableIRQRequest(IRQn_Type irq, uint8_t priority)
{
    NVIC_SetPriority(irq, priority);

    NVIC_EnableIRQ(irq);
}

/*!
 * @brief       Disable NVIC request
 *
 * @param       irq:    The NVIC interrupt request, detailed in IRQn_Type
 *
 * @retval      None
 */
void NVIC_DisableIRQRequest(IRQn_Type irq)
{
    NVIC_DisableIRQ(irq);
}

/**
 * @brief       Enables the system to enter low power mode
 *
 * @param       lowPowerMode: Specifies the system to enter low power mode
 *                      This parameter can be one of the following values:
 *                      @arg NVIC_LOWPOER_SEVONPEND:   Low Power SEV on Pend
 *                      @arg NVIC_LOWPOER_SLEEPDEEP:   Low Power DEEPSLEEP request
 *                      @arg NVIC_LOWPOER_SLEEPONEXIT: Low Power Sleep on Exit
 *
 * @retval      None
 */
void NVIC_EnableSystemLowPower(uint8_t lowPowerMode)
{
    SCB->SCR |= lowPowerMode;
}

/**
 * @brief       Disables the system to enter low power mode
 *
 * @param       lowPowerMode: Specifies the system to enter low power mode
 *                      This parameter can be one of the following values:
 *                      @arg NVIC_LOWPOER_SEVONPEND:   Low Power SEV on Pend
 *                      @arg NVIC_LOWPOER_SLEEPDEEP:   Low Power DEEPSLEEP request
 *                      @arg NVIC_LOWPOER_SLEEPONEXIT: Low Power Sleep on Exit
 *
 * @retval      None
 */
void NVIC_DisableSystemLowPower(uint8_t lowPowerMode)
{
    SCB->SCR &= (uint32_t)(~(uint32_t)lowPowerMode);
}

/**
 * @brief       Configures the SysTick clock source
 *
 * @param       sysTickCLKSource: specifies the SysTick clock source
 *                     This parameter can be one of the following values:
 *                     @arg SYSTICK_CLKSOURCE_HCLK_DIV8: AHB clock divided by 8 selected as SysTick clock source
 *                     @arg SYSTICK_CLKSOURCE_HCLK:      AHB clock selected as SysTick clock source
 *
 * @retval      None
 */
void SysTick_ConfigClockSource(uint32_t sysTickCLKSource)
{
    if (sysTickCLKSource == SYSTICK_CLKSOURCE_HCLK)
    {
        SysTick->CTRL |= (uint32_t)SYSTICK_CLKSOURCE_HCLK;
    }
    else
    {
        SysTick->CTRL &= (~(uint32_t)SYSTICK_CLKSOURCE_HCLK);
    }
}

/*!
 * @brief       Enter Wait Mode
 *
 * @param       None
 *
 * @retval      None
 */
void PMU_EnterWaitMode(void)
{
    SCB->SCR &= (uint32_t)(~(uint32_t)NVIC_LOWPOWER_SLEEPDEEP);
    __WFI();
}

/*!
 * @brief       Enter Stop Mode with WFI instruction
 *
 * @param       None
 *
 * @retval      None
 */
void PMU_EnterHaltModeWFI(void)
{
    SCB->SCR |= NVIC_LOWPOWER_SLEEPDEEP;
    __DSB();
    __WFI();
}

/*!
 * @brief       Enter Stop Mode with WFE instruction
 *
 * @param       None
 *
 * @retval      None
 */
void PMU_EnterHaltModeWFE(void)
{
    SCB->SCR |= NVIC_LOWPOWER_SLEEPDEEP;
    __DSB();
    __WFE();
}

/**@} end of group MISC_Fuctions*/
/**@} end of group MISC_Driver */
/**@} end of group Peripherals_Library*/
